// IGCAPI.C
//
// Application programming interface for IGC download, conversion,
// and validation DLLs.
//
// Copyright (c) 2000-2001 Marc Ramsey. All rights reserved.
//
// http://www.ranlog.com/ramsey/
//
// This software is provided as is with no warranties of any kind, 
// including the warranties of design, merchantibility and fitness 
// for a particular purpose, noninfringement, or arising from a 
// course of dealing, usage or trade practice.  This software is 
// provided with no support and without any obligation to assist in 
// its use, correction, modification, or enhancement.
//
// Permission to use, copy, or modify this software for any purpose 
// is hereby granted without fee, provided the above notices are 
// retained on all copies of the source code. 

#include <windows.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <TCHAR.H>

#define IGCAPIDLL_EXPORT
#include "IGCAPI.H"

typedef DWORD (*ProcIdentifyDLL)(LPTSTR value, DWORD size);
typedef HICON (*ProcLoadIconDLL)();
typedef VOID (*ProcInitializeDLL)(HWND windowHandle, BOOL quietMode);
typedef VOID (*ProcSetWindowDLL)(HWND windowHandle);
typedef DWORD (*ProcKeepAwakeIntervalDLL)();
typedef BOOL  (*ProcUseSerialOptionsDLL)();
typedef DWORD (*ProcSerialOptionsDLL)(LPTSTR options, DWORD size);
typedef BOOL (*ProcSerialConnectFR)(LPCTSTR device, LPCTSTR options);
typedef BOOL (*ProcKeepAwakeFR)();
typedef DWORD (*ProcIdentifyFR)(LPTSTR value, DWORD size);
typedef DWORD (*ProcIdentifyLogFR)(DWORD index, LPTSTR value, DWORD size);
typedef BOOL (*ProcDownloadLogFR)(DWORD index, LPCTSTR filename);
typedef VOID (*ProcDisconnectFR)();
typedef BOOL (*ProcUseConvertLog)();
typedef BOOL (*ProcConvertLog)(LPCTSTR filename, LPCTSTR igcFilename);
typedef BOOL (*ProcValidateLog)(LPCTSTR filename);

typedef struct {
	HMODULE module;

	ProcIdentifyDLL IdentifyDLL;
	ProcLoadIconDLL LoadIconDLL;
	ProcInitializeDLL InitializeDLL;
	ProcSetWindowDLL SetWindowDLL;
	ProcKeepAwakeIntervalDLL KeepAwakeIntervalDLL;
	ProcUseSerialOptionsDLL UseSerialOptionsDLL;
	ProcSerialOptionsDLL SerialOptionsDLL;
	ProcSerialConnectFR SerialConnectFR;
	ProcKeepAwakeFR KeepAwakeFR;
	ProcIdentifyFR IdentifyFR;
	ProcIdentifyLogFR IdentifyLogFR;
	ProcDownloadLogFR DownloadLogFR;
	ProcDisconnectFR DisconnectFR;
	ProcUseConvertLog UseConvertLog;
	ProcConvertLog ConvertLog;
	ProcValidateLog ValidateLog;
} IGCAPI;

VOID IGC_API UnloadAllDLLs();

static HWND windowHandle;
static IGCAPI *activeDLL;
static IGCAPI **loadedDLLs;
static DWORD numLoadedDLLs;

// DllMain
//
// This is called when DLL is first loaded
//
BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
	activeDLL = NULL;
	loadedDLLs = NULL;
    return TRUE;
}

// ErrorBox
//
// Display the supplied error message string in a 
// modal dialog box.
//
VOID IGC_API ErrorBox(LPCTSTR message)
{
	MessageBox(windowHandle, (LPCTSTR)message, "Error", MB_OK | MB_ICONEXCLAMATION );
}

// LastErrorBox
//
// Display the last detected error in a modal dialog
//
VOID IGC_API LastErrorBox(LPCTSTR arg, ...)
{
	va_list marker;
	LPVOID lpMsgBuf;

	va_start(marker, arg);

	//
	// Format error message
	//
	FormatMessage( 
		FORMAT_MESSAGE_ALLOCATE_BUFFER | 
		FORMAT_MESSAGE_FROM_SYSTEM,
		NULL,
		GetLastError(),
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
		(LPTSTR) &lpMsgBuf,
		0,
		&marker 
	);

	//
	// Display the dialog
	//
	ErrorBox((LPCTSTR)lpMsgBuf);
	// Free the buffer.
	LocalFree( lpMsgBuf );
	va_end(marker);
}

static DWORD IGC_API GetField(int field, LPTSTR buffer, LPTSTR value, DWORD size)
{
	TCHAR *ptr = _tcstok(buffer, _T("|"));

	while (field-- > 0)
		ptr = _tcstok(NULL, _T("|"));

	if (ptr) {
		_tcsncpy(value, ptr, size);
		value[size - 1] = '\0';
	} else
		*value = '\0';
	return _tcslen(value);
}

DWORD IGC_API IdentifyDLLField(IdDLLField field, LPTSTR value, DWORD size)
{
	TCHAR buffer[256];
	IdentifyDLL(buffer, sizeof(buffer));
	return GetField((int)field, buffer, value, size);
}

DWORD IGC_API IdentifyDLLExt(IdDLLField field, LPTSTR string, DWORD size)
{
	TCHAR buffer[128], buffer2[256], *ptr;

	IdentifyDLLField(field, buffer, sizeof(buffer));
	ptr = _tcstok(buffer, _T(","));
	*buffer2 = '\0';

	while (ptr) {
		if (*buffer2)
			_tcscat(buffer2, _T(";"));
		_tcscat(buffer2, _T("*."));
		_tcscat(buffer2, ptr);
		ptr = _tcstok(NULL, _T(","));
	}
	_stprintf(string, "Log Files (%s)|%s|All Files(*.*)|*.*||", buffer2, buffer2);
	return _tcslen(string);
}

LPVOID IGC_API LoadDLL(LPCTSTR filename)
{
	HMODULE mod = LoadLibrary(filename);
	IGCAPI *api = NULL;

	// If there is an active DLL, make sure it is disconnected
	if (activeDLL)
		activeDLL->DisconnectFR();
	activeDLL = NULL;

	// Get pointers to functions
	if (mod && (api = malloc(sizeof(IGCAPI)))
	&& (api->IdentifyDLL = (ProcIdentifyDLL) GetProcAddress(mod, "IdentifyDLL"))
	&& (api->LoadIconDLL = (ProcLoadIconDLL) GetProcAddress(mod, "LoadIconDLL"))
	&& (api->InitializeDLL = (ProcInitializeDLL) GetProcAddress(mod, "InitializeDLL"))
	&& (api->SetWindowDLL = (ProcSetWindowDLL) GetProcAddress(mod, "SetWindowDLL"))
	&& (api->KeepAwakeIntervalDLL = (ProcKeepAwakeIntervalDLL) GetProcAddress(mod, "KeepAwakeIntervalDLL"))
	&& (api->UseSerialOptionsDLL = (ProcUseSerialOptionsDLL) GetProcAddress(mod, "UseSerialOptionsDLL"))
	&& (api->SerialOptionsDLL = (ProcSerialOptionsDLL) GetProcAddress(mod, "SerialOptionsDLL"))
	&& (api->SerialConnectFR = (ProcSerialConnectFR) GetProcAddress(mod, "SerialConnectFR"))
	&& (api->KeepAwakeFR = (ProcKeepAwakeFR) GetProcAddress(mod, "KeepAwakeFR"))
	&& (api->IdentifyFR = (ProcIdentifyFR) GetProcAddress(mod, "IdentifyFR"))
	&& (api->IdentifyLogFR = (ProcIdentifyLogFR) GetProcAddress(mod, "IdentifyLogFR"))
	&& (api->DownloadLogFR = (ProcDownloadLogFR) GetProcAddress(mod, "DownloadLogFR"))
	&& (api->DisconnectFR = (ProcDisconnectFR) GetProcAddress(mod, "DisconnectFR"))
	&& (api->UseConvertLog = (ProcUseConvertLog) GetProcAddress(mod, "UseConvertLog"))
	&& (api->ConvertLog = (ProcConvertLog) GetProcAddress(mod, "ConvertLog"))
	&& (api->ValidateLog = (ProcValidateLog) GetProcAddress(mod, "ValidateLog"))) {
		api->module = mod;
		activeDLL = api;
		return (LPVOID)api;
	} else {
		LastErrorBox(filename);
		if (api)
			free(api);
		return NULL;
	}
}

VOID IGC_API UnloadDLL(LPVOID igcdll)
{
	IGCAPI *api = (IGCAPI*)igcdll;
	if (api) {
		// Make sure serial port is disconnected;
		api->DisconnectFR();
		// If active DLL, clear active pointer
		if (api == activeDLL)
			activeDLL = NULL;
		// Free library and API structure
		FreeLibrary(api->module);
		free(api);
	}
}

DWORD IGC_API LoadAllDLLs(LPCTSTR directory)
{
	WIN32_FIND_DATA file;
	HANDLE hFind;
	BOOL more = TRUE;
	DWORD pathSize, count = 0;
	char path[MAX_PATH], pattern[MAX_PATH], *ptr;

	// If DLLs loaded already, unload them
	if (loadedDLLs)
		UnloadAllDLLs();

	//
	// Create file search pattern
	//

	if (directory) {
		// Directory specified, copy to path and append backslash
		strncpy(path, directory, sizeof(path) - 1);
		path[sizeof(path) - 2] = 0;
		pathSize = strlen(path);
		if (path[pathSize - 1] != '\\') {
			path[pathSize++] = '\\';
			path[pathSize] = 0;
		}
	} else {
		// No path supplied, get path for executable module
		pathSize = GetModuleFileName(NULL, path, sizeof(path));
		path[sizeof(path) - 1] = 0;
		if (ptr = strrchr(path, '\\')) {
			*++ptr = 0;
			pathSize = ptr - path;
		} else {
			path[0] = 0;	// if all else fails, try current directory
			pathSize = 0;
		}
	}
	// Create pattern string
	strcpy(pattern, path);
	strncat(pattern + pathSize, "igc-*.dll", sizeof(pattern) - pathSize);

	//
	// Count the number of DLLs available
	//

	hFind = FindFirstFile(pattern, &file);
	if (hFind == INVALID_HANDLE_VALUE) {
		if (GetLastError() != ERROR_NO_MORE_FILES) 
			LastErrorBox(pattern);
		return 0;
	}

	while (more) {
		count += 1;
		more = FindNextFile(hFind, &file);
	}

	FindClose(hFind);

	//
	// Load the DLLs
	//

	loadedDLLs = (IGCAPI**)malloc(count*sizeof(IGCAPI*));
	hFind = FindFirstFile(pattern, &file);
	more = TRUE;

	while (more && numLoadedDLLs < count) {
		strncpy(path + pathSize, file.cFileName, sizeof(path) - pathSize);
		loadedDLLs[numLoadedDLLs] = (IGCAPI*)LoadDLL(path);
		if (loadedDLLs[numLoadedDLLs++] == NULL)
			numLoadedDLLs--;
		more = FindNextFile(hFind, &file);
	}

	FindClose(hFind);
	return numLoadedDLLs;
}

BOOL IGC_API SelectDLL(DWORD index)
{
	if (loadedDLLs && index < numLoadedDLLs) {
		activeDLL = loadedDLLs[index];
		return TRUE;
	} else {
		activeDLL = NULL;
		return FALSE;
	}
}

VOID IGC_API UnloadAllDLLs()
{
	DWORD i;

	for (i = 0; i < numLoadedDLLs; i++)
		UnloadDLL(loadedDLLs[i]);

	free(loadedDLLs);
	loadedDLLs = NULL;
	numLoadedDLLs = 0;
}

DWORD IGC_API IdentifyDLL(LPTSTR value, DWORD size)
{
	*value = '\0';
	return activeDLL? activeDLL->IdentifyDLL(value, size) : 0;
}

HICON IGC_API LoadIconDLL()
{
	return activeDLL ? activeDLL->LoadIconDLL() : (HICON)0;
}

VOID IGC_API InitializeDLL(HWND windowHandle, BOOL quietMode)
{
	if (activeDLL)
		activeDLL->InitializeDLL(windowHandle, quietMode);
}

VOID IGC_API SetWindowDLL(HWND windowHandle)
{
	if (activeDLL)
		activeDLL->SetWindowDLL(windowHandle);
}

DWORD IGC_API KeepAwakeIntervalDLL()
{
	return activeDLL ? activeDLL->KeepAwakeIntervalDLL() : 0;
}

BOOL IGC_API UseSerialOptionsDLL()
{
	return activeDLL? activeDLL->UseSerialOptionsDLL() : FALSE;
}

DWORD IGC_API SerialOptionsDLL(LPTSTR options, DWORD size)
{
	if (activeDLL && activeDLL->SerialOptionsDLL)
		return activeDLL->SerialOptionsDLL(options, size);
	else
		return -2;
}

BOOL IGC_API SerialConnectFR(LPCTSTR device, LPCTSTR options)
{
	return activeDLL ? activeDLL->SerialConnectFR(device, options) : FALSE;
}

BOOL IGC_API KeepAwakeFR()
{
	return activeDLL ? activeDLL->KeepAwakeFR() : FALSE;
}

DWORD IGC_API IdentifyFR(LPTSTR value, DWORD size)
{
	*value = '\0';
	return activeDLL ? activeDLL->IdentifyFR(value, size) : 0;
}

DWORD IGC_API IdentifyLogFR(DWORD index, LPTSTR value, DWORD size)
{
	*value = '\0';
	return activeDLL ? activeDLL->IdentifyLogFR(index, value, size) : 0;
}

BOOL IGC_API DownloadLogFR(DWORD index, LPCTSTR filename)
{
	return activeDLL ? activeDLL->DownloadLogFR(index, filename) : FALSE;
}

VOID IGC_API DisconnectFR()
{
	if (activeDLL)
		activeDLL->DisconnectFR();
}

BOOL IGC_API UseConvertLog()
{
	return activeDLL ? activeDLL->UseConvertLog() : FALSE;
}

BOOL IGC_API ConvertLog(LPCTSTR filename, LPCTSTR igcFilename)
{
	if (activeDLL && activeDLL->ConvertLog)
		return activeDLL->ConvertLog(filename, igcFilename);
	else
		return FALSE;
}

BOOL IGC_API ValidateLog(LPCTSTR filename)
{
	return activeDLL ? activeDLL->ValidateLog(filename) : FALSE;
}
